from django.shortcuts import render
from rest_framework import viewsets, serializers, status
from rest_framework.permissions import IsAuthenticated
from rest_framework.decorators import action
from items.item_serializer import ItemSerializer, UnitySerializer
from items.models import Item, Unity
from django.http import JsonResponse

# Create your views here.

class ItemViewSet(viewsets.ModelViewSet):
    unity = serializers.ReadOnlyField(source='unity.label')
    queryset = Item.objects.all()
    serializer_class = ItemSerializer
    permission_classes = [IsAuthenticated]

    def get_queryset(self):
        """
        This view should return a list of all the purchases
        for the currently authenticated user.
        """
        user = self.request.user
        return Item.objects.filter(user=user)
    def update(self, request, *args, **kwargs):
        """
        Update an existing item for the authenticated user.
        """
        partial = kwargs.pop('partial', False)  # To support partial updates
        instance = self.get_object()  # Get the specific instance being updated
        serializer = self.get_serializer(instance, data=request.data, partial=partial)
        
        # Validate data
        serializer.is_valid(raise_exception=True)
        # Save updates
        serializer.save()
        return JsonResponse(serializer.data, status=status.HTTP_200_OK)

    def partial_update(self, request, *args, **kwargs):
        """
        Partially update an existing item for the authenticated user.
        """
        kwargs['partial'] = True
        return self.update(request, *args, **kwargs)
    
    @action(detail=False, methods=['POST'])
    def save_multiple_items(self, request):
        """
        This view should save items from a JSON array to the database
        for the currently authenticated user.
        """
        user = self.request.user
        data = request.data.get('items')
        if not data:
            return Response({"error": "No items found in the JSON array."}, status=status.HTTP_400_BAD_REQUEST)
        else:
            for item in data:
                Item.objects.create(
                    user=user,
                    label=item['label'],
                    price=item['price'],
                    unit=item['unit']
                )
            return JsonResponse({"success": "Items saved successfully."})


class UnityViewSet(viewsets.ModelViewSet):
    queryset = Unity.objects.all()
    serializer_class = UnitySerializer
    permission_classes = [IsAuthenticated]    
